﻿
#Params
# $ZabbixAgentDist
# $Path
# $Config
# $Firewall yes or no
# $Reinstall

#Functions
# Get-SysArch
# Create-Config #if $Config is not defined
# Get-Archive
# Unzip-Archive if $Reinstall is yes then rename old and rewrite
# Install-Zabbix
# Add-FirewallRules
# Write-Log
# Add some checks tcp servers port availability

function Get-zbaSysArch
    {
 
    Param(
     
    )
    # End of Parameters
 
    #This function returns win64 or win32 value depends system architecture
    $SysArch = wmic os get OSArchitecture /value
        if ($SysArch -eq "OSArchitecture=64-bit")
            {
                $SysArch = 'win64'
            }
        else
            {
                $SysArch = 'win32'
            }
        return $SysArch
     }

function Remove-zbaService
  {
    [cmdletbinding()]
    
    param(     
         )
   # End of Parameters
   
   Try 
       {
         if (Get-service -name "Zabbix Agent" -ErrorAction SilentlyContinue)
           {
             $confirmation = Read-Host "Zabbix Agent Service Already installed!`nRemove it? [y/n]"
               if ($confirmation -eq 'y') 
                 {
                   Write-host "Zabbix Agent Service Removing..."
                   Get-service -name "Zabbix Agent" | Stop-Service
                   (Get-WmiObject Win32_Service -filter "name='Zabbix Agent'").Delete() | out-null
                   Start-sleep 3
                 }
               else
                  {
                   Exit(0)
                  }
             
           }
       }
   Catch
       {
         $ErrorMessage = $_.Exception.Message
         $FailedItem = $_.Exception.ItemName
         write-host "Cannot remove Zabbix Agent Service"
         write-host $ErrorMessage
         Break
       }
  }


function New-zbaConfig
  {
    [cmdletbinding()]
    param(
           [string[]] $MonitoringServer, 
           [string[]] $Path
         )
   # End of Parameters
   # Getting hostname
   $CompHostName ="$env:computername.$env:userdnsdomain"
   # Removing dot from end of the hostname (dot appears when computer is not a domain member)
   $CompHostName = $CompHostName -replace "\.(?<=\z)", ""
   $ConfContent = "
LogFile=$Path\log\zabbix_agentd.log
Server=$MonitoringServer 
ServerActive=$MonitoringServer 
Hostname=$CompHostName
HostMetadataItem=system.uname
                 "
   Return $ConfContent
  }



function Get-zbaArchive
  {
    #Requires -Module BitsTransfer
    [cmdletbinding()]
    param(
           [string[]] $ZabbixAgentDist, 
           [string[]] $TempArchive = "$env:TEMP\zabbix_agent.zip"
         )
   # End of Parameters
   Import-Module BitsTransfer -ErrorAction stop
   Try 
       {
         Start-BitsTransfer -Source $ZabbixAgentDist -Destination $TempArchive -ErrorAction stop
       }
   Catch
       {
         $ErrorMessage = $_.Exception.Message
         $FailedItem = $_.Exception.ItemName
         write-host "Cannot download\copy Zabbix Agent archive from $ZabbixAgentDist"
         write-host $ErrorMessage
         Break
       }
  }
  

  function Unzip-zbaArchive
  {
    [cmdletbinding()]
    param(
           [string[]] $Path, 
           [string[]] $TempArchive = "$env:TEMP\zabbix_agent.zip"
         )
   # End of Parameters
    if(!(Test-Path -Path $Path)) # If Zabbix folder not exists,
                                 # then extract files
      {
        Add-Type -AssemblyName System.IO.Compression.FileSystem
        [System.IO.Compression.ZipFile]::ExtractToDirectory($TempArchive, $Path)
        New-Item -ItemType directory -Path $Path\log -ErrorAction Stop | out-null
      }
    else                        # If Zabbix folder exists rename in 
                                # in ZabbixOLD folder
                                # Service should be stopped and removed by
                                # Remove-zbaService function
      {
        Try
           {
        Rename-Item -Path "$Path" -NewName "ZabbixOLD" -Force
        Add-Type -AssemblyName System.IO.Compression.FileSystem
        [System.IO.Compression.ZipFile]::ExtractToDirectory($TempArchive, $Path)
        New-Item -ItemType directory -Path $Path\log -ErrorAction Stop | out-null
        Remove-Item -Force $TempArchive
           }
           Catch
       {
         $ErrorMessage = $_.Exception.Message
         $FailedItem = $_.Exception.ItemName
         write-host "Cannot Rename $Path`nPlease rename it manually!"
         write-host $ErrorMessage
         Break
       }
      }
     
  }
  
 function Write-zbaConfig
  {
  # Just run New-zbaConfig function with parameter MonitoringServer
  # and write result to the config file
    [cmdletbinding()]
    param(
           [string[]] $Path,
           [string[]] $MonitoringServer
         )
   # End of Parameters
   # Getting hostname
    Set-Content -Value (New-zbaConfig -MonitoringServer $MonitoringServer -Path $Path) -Path "$Path\conf\zabbix_agentd.win.conf"
  }

function Install-zbaService
  {
    # Installing Zabbix Agent as a Service
    # This function uses Get-zbaSysArch for 
    # determination system architecture win32 or win64
    [cmdletbinding()]
    param(
           [string[]] $Path,
           [string[]] $SysArch
           
         )
   # End of Parameters
   $ZabbixInstallCommand = "`"$Path\bin\$SysArch\zabbix_agentd.exe`" --config `"$Path\conf\zabbix_agentd.win.conf`""

   # Installing Zabbix Service
   Try 
     {
       New-Service -Name "Zabbix Agent" -BinaryPathName $ZabbixInstallCommand `
       -DisplayName "Zabbix Agent" -Description "Provides Monitoring" -StartupType "Automatic" | out-null
     }
   Catch
     {
      $ErrorMessage = $_.Exception.Message
      $FailedItem = $_.Exception.ItemName
      write-host "Cannot Install Zabbix Service!!!"
      write-host $ErrorMessage
      Break
     }
  }


function Set-zbaFirewallRules
  {
    [cmdletbinding()]
    param(
           [string[]] $MonitoringServer
         )
   # End of Parameters
   #Adding firewall rules for Zabbix Agent and Zabbix Server
   $MonServerIp = (Resolve-DNSName "$MonitoringServer").IP4Address
   Try
      {
        New-NetFirewallRule -DisplayName "Zabbix Agent IN" -RemoteAddress $MonServerIp `
        -Profile Any -Action Allow -Direction Inbound -Protocol TCP -LocalPort 10050 | out-null
        New-NetFirewallRule -DisplayName "Zabbix Agent OUT" -RemoteAddress $MonServerIp `
        -Profile Any -Action Allow -Direction OutBound -Protocol TCP -LocalPort 10051 | out-null
      }
   Catch
      {
        $ErrorMessage = $_.Exception.Message
        $FailedItem = $_.Exception.ItemName
        write-host "Cannot Add Firewall Rules..."
        write-host $ErrorMessage
        Break
      }
  }




function Install-ZabbixAgent
  {
 [CmdletBinding()]
  param
  (
    [Parameter(Mandatory=$True,
               HelpMessage='What path for Zabbix Agent folder do you prefer?')]
    
    [string[]]$Path,
	[Parameter(Mandatory=$True,
               HelpMessage='What is your Zabbix Server Name?')]
    
    [string]$Server,
    [Parameter(Mandatory=$True,
               HelpMessage='Where is your zabbix agent zip archive?')]
    
    [string]$ZabbixDist
  )
New-zbaConfig -MonitoringServer "$Server" -Path "$Path"
Get-zbaArchive -ZabbixAgentDist $ZabbixDist
Remove-zbaService
Unzip-zbaArchive -Path "$Path"
Write-zbaConfig -Path "$Path" -MonitoringServer $Server
Install-zbaService -Path "$Path" -SysArch (Get-zbaSysArch)
Set-zbaFirewallRules -MonitoringServer "$Server"
Start-service -name "Zabbix Agent"

For ($i=5; $i -gt 1; $i–-) {  
    Write-Progress -Activity "`nInstallation completed! Waiting Zabbix Agent starts..." `
     -SecondsRemaining $i
    Start-Sleep 1
}

Get-content $Path\log\zabbix_agentd.log

  }

Install-ZabbixAgent -Server monitoring.nugaev.net `
-ZabbixDist "https://support.nugaev.net/files/distrib/zabbix_agent.zip" `
-Path "C:\Program Files\Zabbix"